#include "dm_os.h"
#include "dm_linux_internal.h"
#include <linux/slab.h>
#include <linux/kthread.h>

struct DM_LINUX_TASK
{
    TASKFUNC entry_point;
    UINT32 *param;
    struct task_struct *task;
    long priority;
};

static int pseudoTask(void *dmTaskData)
{
    int sched_err;
    char sched_name_buf[TASK_COMM_LEN];
    struct sched_param param;

    struct DM_LINUX_TASK *task = (struct DM_LINUX_TASK*) dmTaskData;
    param.sched_priority = task->priority;
	/* We need realtime guarantees, real-time for everyone! */
    sched_err = sched_setscheduler(current, SCHED_RR, &param);
    get_task_comm(sched_name_buf, current);
    if(sched_err) {
        pr_err("=========== Failed to set priority to %i =========== task = %i name = %s err = %i\n",
                param.sched_priority,
                current->pid,
                sched_name_buf,
                sched_err);
    }
//    /* Subtract 20 to convert the priority to the nice range -20 -> 0 -> 20. */
//    set_user_nice(current, (task->priority-20));
    (task->entry_point)((UINT32) task->param);
    return 0;
}

/* TASKFUNC looks like this: typedef void (*TASKFUNC)(UINT32 param); */
UINT32 OsCreateNamedAdvTaskPri(TASKFUNC entryPoint, UINT16 usStackDepth, UINT32 * pParam, UINT32 uxPriority,
        const signed char * const pName)
{
    /* TODO: usStackDepth is not a priority to implement. */
    struct DM_LINUX_TASK *pseudo_task_data = kzalloc(sizeof(struct DM_LINUX_TASK), GFP_KERNEL);
    if(!pseudo_task_data) {
        pr_err("Failed to allocate memory for kernel thread data.\n");
        return 0;
    }
    /* setup function call */
    pseudo_task_data->entry_point = entryPoint;
    pseudo_task_data->param = pParam;
    /* DM interface exposes priority as unsigned, this is signed but much
       larger than expected values. */
    pseudo_task_data->priority = uxPriority;
    /* Start thread. Priority is set from inside the created thread. */
    pseudo_task_data->task = kthread_run(pseudoTask, pseudo_task_data, pName);
    if(IS_ERR(pseudo_task_data->task)) {
        pr_err("Failed to start thread %s\n", pName);
        kfree(pseudo_task_data);
        pseudo_task_data = 0;
    }
    return (DM_RESOURCE_HANDLE)pseudo_task_data;
}

void HwDelayMsec(UINT32 ms)
{
    msleep(ms);
}

UINT32 HwGetMsec(void)
{
    UINT32 ms;
    struct timespec ts;
    getrawmonotonic(&ts);
    ms = (((s64)ts.tv_sec) * MSEC_PER_SEC) + (ts.tv_nsec / NSEC_PER_MSEC);
    return ms;
}


static struct mutex critical_mutex;
/**
 * Initialize any global parameters that the dm_os is dependent on.
 */
void OsStart(void)
{
    mutex_init(&critical_mutex);
}

/* Locking functions */
void OsDeleteLock(DM_RESOURCE_HANDLE lHandle)
{
    struct mutex *m = (struct mutex*) lHandle;
    if(!m) {
        pr_err("Cannot delete null lock\n");
    }
    kfree(m);
}
void OsCreateLock(DM_RESOURCE_HANDLE *pHandle)
{
    struct mutex *m = kzalloc(sizeof(struct mutex), GFP_KERNEL);
    if(!m) {
        pr_err("Failed to allocate mutex\n");
    }
    else {
        mutex_init(m);
        (*pHandle) = (DM_RESOURCE_HANDLE)m;
    }
}
INT32 OsLock(UINT32 handle)
{
    struct mutex *m = (struct mutex*)handle;
    if(!m) {
        pr_err("Cannot lock null mutex\n");
        return -1;
    }
    mutex_lock(m);
    return 0;
}
//INT32 OsLockWait(UINT32 handle, UINT32 waitTicks)
//{
//    struct mutex *m = (struct mutex*)handle;
//    if(!m) {
//        pr_err("Cannot lock null mutex\n");
//    }
//    mutex_lock(m);
//}
INT32 OsUnlock(UINT32 handle)
{
    struct mutex *m = (struct mutex*)handle;
    if(!m) {
        pr_err("Cannot unlock null mutex\n");
        return -1;
    }
    mutex_unlock(m);
    return 0;
}

/**
 * @param itemCnt How many items can the queue store
 * @param lItemSize The size of each item
 * @return handle to the fifo
 */
DM_RESOURCE_HANDLE OsQueueCreate(UINT32 itemCnt, UINT32 lItemSize)
{
    int ret = 0;
    struct LinuxQueue *queue = kzalloc(sizeof(struct LinuxQueue), GFP_KERNEL);
    size_t queueSize = roundup_pow_of_two(lItemSize*itemCnt);
    if(checkQueue(queue)) {
        /* Allocation failed, return 0 handle */
        return 0;
    }
    ret = kfifo_alloc(&queue->fifo, queueSize, GFP_KERNEL);
    if(ret) {
        /* Allocation failed, return 0 handle */
        kfree(queue);
        return 0;
    }
    init_waitqueue_head(&queue->wq);
    spin_lock_init(&queue->slock);
    queue->objectSize = lItemSize;

    //PRINT_QUEUE(queue);
    /* Return the pointer as the handle. */
    return (DM_RESOURCE_HANDLE) (queue);
}

/* Why does this use INT32 as the handle? */
void OsQueueDelete(INT32 handle)
{
    struct LinuxQueue *queue = (struct LinuxQueue*) (handle);
    //PRINT_QUEUE(queue);
    if(checkQueue(queue)) {
        printk(KERN_ERR "Cannot delete pointer to zero queue, handle=%p\n", queue);
        return;
    }
    kfifo_free(&queue->fifo);
    kfree(queue);
}

/**
 * Not as useful without knowing a specific queue to clear.
 */
void OsQueueClear(void)
{
    pr_warn("Because go fuck yourself.\n");
}

INT32 OsQueue(DM_RESOURCE_HANDLE handle, void* item, UINT32 timeout)
{
    /* TBD: Not sure what to do with the timeout...? */
    struct LinuxQueue *queue = (struct LinuxQueue*) (handle);
    //PRINT_QUEUE(queue);
    if(checkQueue(queue) || !item) {
        printk(KERN_ERR "Cannot queue item, pointer is zero. handle=%p item=%p\n", queue, item);
        return -1;
    }
    //return kfifo_in(&queue->fifo, item, queue->objectSize);
    return kfifo_in_spinlocked(&queue->fifo, item, queue->objectSize, &queue->slock);
}

INT32 OsDequeue(DM_RESOURCE_HANDLE handle, void* item)
{
    struct LinuxQueue *queue = (struct LinuxQueue*) (handle);
    //PRINT_QUEUE(queue);
    if(checkQueue(queue) || !item) {
        printk(KERN_ERR "Cannot dequeue item, pointer is zero. handle=%p item=%p\n", queue, item);
        return -1;
    }
    //return kfifo_out(&queue->fifo, item, queue->objectSize);
    return kfifo_out_spinlocked(&queue->fifo, item, queue->objectSize, &queue->slock);
}

//TODO: Remove function altogether, write isEmpty, isFull.
//It is used as a check to see if any items exist, but it is not very accurate.
UINT32 OsGetQueueCount(DM_RESOURCE_HANDLE handle)
{
    UINT32 bytesUsed;
    unsigned long flags;
    struct LinuxQueue *queue = (struct LinuxQueue*) (handle);
    //PRINT_QUEUE(queue);
    if(checkQueue(queue)) {
        //BUG_ON(!queue);
        printk(KERN_ERR "Cannot get queue count, pointer is zero. handle=%p\n", queue);
        return 0;
    }
    /* This only returns approximate size, since some objects in the queue
     * can be larger than others. */
    spin_lock_irqsave(&queue->slock, flags);
    bytesUsed = kfifo_len(&queue->fifo);
    spin_unlock_irqrestore(&queue->slock, flags);
    return bytesUsed;
    //return bytesUsed / queue->objectSize;
}

/**
 * Event Group Related Functions
 * No destroy function!
 */
DM_RESOURCE_HANDLE OsCreateEventGroup(void)
{
    struct LinuxEvent *le = kzalloc(sizeof(*le), GFP_KERNEL);
    if(!le) {
        pr_err("Failed to create Event.\n");
        return 0;
    }
    le->type = DM_LINUX_EVENT_TYPE_GROUP;
    init_waitqueue_head(&le->wq);
    //mutex_init(&le->m);
    spin_lock_init(&le->slock);
    le->events = 0;
    return (DM_RESOURCE_HANDLE) le;
}
UINT32 OsSetEvent(DM_RESOURCE_HANDLE eventHandle, UINT32 mask)
{
    unsigned long flags;
    struct LinuxEvent *le = (struct LinuxEvent*) (eventHandle);
    if(eventErrors(le, DM_LINUX_EVENT_TYPE_GROUP)) {
        pr_err("Cannot set null event or wrong event type.\n");
        return false;
    }
//    mutex_lock(&le->m);
    spin_lock_irqsave(&le->slock, flags);
    le->events |= mask;
    //    mutex_unlock(&le->m);
    spin_unlock_irqrestore(&le->slock, flags);

    wake_up_all(&le->wq);
    return 0;
}
UINT32 OsWaitForEventAndClear(DM_RESOURCE_HANDLE eventHandle, UINT32 mask)
{
    unsigned long flags;
    struct LinuxEvent *le = (struct LinuxEvent*) (eventHandle);
    UINT32 events = 0;
    if(eventErrors(le, DM_LINUX_EVENT_TYPE_GROUP)) {
        pr_err("Cannot delete null event or wrong event type.\n");
        return 0;
    }
    wait_event(le->wq, le->events & mask);
    /* we woke up, saw our events, clear them. */
//    mutex_lock(&le->m);
    spin_lock_irqsave(&le->slock, flags);
    events = le->events & mask;
    le->events &= ~events;
    //    mutex_unlock(&le->m);
    spin_unlock_irqrestore(&le->slock, flags);

    return events;
}

BOOL OsCreateEvent(DM_RESOURCE_HANDLE* eventHandle)
{
    struct LinuxEvent *le = kzalloc(sizeof(*le), GFP_KERNEL);
    if(!le) {
        pr_err("Failed to create Event.\n");
        return 0;
    }
    le->type = DM_LINUX_EVENT_TYPE_SINGLE;
    init_waitqueue_head(&le->wq);
//    mutex_init(&le->m);
    spin_lock_init(&le->slock);
    le->events = 0;
    (*eventHandle) = (DM_RESOURCE_HANDLE) le;
    return true;
}

void OsDeleteEvent(DM_RESOURCE_HANDLE eventHandle)
{
    struct LinuxEvent *le = (struct LinuxEvent*) (eventHandle);
    if(le == 0) {
        pr_err("Cannot delete null event.\n");
        return;
    }
    kfree(le);
}

void OsSignalEvent(DM_RESOURCE_HANDLE eventHandle)
{
    unsigned long flags;
    struct LinuxEvent *le = (struct LinuxEvent*) (eventHandle);
    if(eventErrors(le, DM_LINUX_EVENT_TYPE_SINGLE)) {
        pr_err("Cannot wait on null Event or wrong Event type.\n");
        return;
    }
//    mutex_lock(&le->m);
    spin_lock_irqsave(&le->slock, flags);
    le->events |= 1;
//    mutex_unlock(&le->m);
    spin_unlock_irqrestore(&le->slock, flags);
    wake_up_all(&le->wq);
}

//INT32 OsSendEvent(UINT32 EventHandle);
BOOL OsWaitEvent(DM_RESOURCE_HANDLE eventHandle, UINT32 waitTime)
{
    unsigned long flags;
    BOOL wasSet = false;
    struct LinuxEvent *le = (struct LinuxEvent*) (eventHandle);
    if(eventErrors(le, DM_LINUX_EVENT_TYPE_SINGLE)) {
        pr_err("Cannot wait on null Event or wrong Event type.\n");
        return false;
    }
    if(waitTime == OS_NO_WAIT) {
//        mutex_lock(&le->m);
        spin_lock_irqsave(&le->slock, flags);
        wasSet = le->events;
        le->events = 0;
//        mutex_unlock(&le->m);
        spin_unlock_irqrestore(&le->slock, flags);
    }
    else if(waitTime == OS_WAIT_FOREVER) {
        wait_event(le->wq, le->events);
//        mutex_lock(&le->m);
        spin_lock_irqsave(&le->slock, flags);
        wasSet = le->events;
        le->events = 0;
//        mutex_unlock(&le->m);
        spin_unlock_irqrestore(&le->slock, flags);
    }
    else {
        wasSet = wait_event_timeout(le->wq, le->events, waitTime * HZ/1000);
        if(wasSet) {
            spin_lock_irqsave(&le->slock, flags);
//            mutex_lock(&le->m);
            le->events = 0;
//            mutex_unlock(&le->m);
            spin_unlock_irqrestore(&le->slock, flags);
        }
    }
    return wasSet;
}

/**
 * The CE interface for this also ignores the val argument, we're doing the same.
 */
DM_RESOURCE_HANDLE OSCreateCountingSemaphore(unsigned val, unsigned initialCount)
{
    struct semaphore *sem = kzalloc(sizeof(*sem), GFP_KERNEL);
    if(!sem) {
        printk(KERN_ERR "Failed to create semaphore.\n");
        return 0;
    }
    sema_init(sem, initialCount);
    return (DM_RESOURCE_HANDLE) sem;
}
UINT32 OSDeleteCountingSemaphore(DM_RESOURCE_HANDLE semId)
{
    struct semaphore *sem = (struct semaphore*) (semId);
    if(!sem) {
        printk(KERN_ERR "Cannot delete null semaphore.\n");
        return -1;
    }
    kfree(sem);
    return 0;
}
UINT32 OSTakeCountingSemaphore(DM_RESOURCE_HANDLE semId, UINT32 waitTime)
{
    struct semaphore *sem = (struct semaphore*) (semId);
    int err;
    if(!sem) {
        printk(KERN_ERR "Cannot delete null semaphore.\n");
        return 0;
    }
    err = down_timeout(sem, waitTime * HZ/1000);
    return !err;
}
UINT32 OSGiveCountingSemaphore(DM_RESOURCE_HANDLE semId)
{
    struct semaphore *sem = (struct semaphore*) (semId);
    if(!sem) {
        printk(KERN_ERR "Cannot delete null semaphore.\n");
        return 0;
    }
    up(sem);
    return 1;
}

/**
 * Linux hrtimers are run in interrupt context, since we don't need that
 * kind of precision and we may want to block, our function will run in a
 * workqueue. Theoretically, a sufficiently small timeout could make the
 * workqueue overflow and the system explode. Be warned.
 */
struct DmLinuxTimer
{
    struct work_struct mywork;
    ktime_t interval;
    void (*func)(void);
    bool periodic;
    struct hrtimer myhrtimer;
};

inline static void printDmLinuxTimer(struct DmLinuxTimer *dm_timer)
{
    pr_info("========== DmLinuxTimer@%p ========== interval=%llins func=%p periodic=%i\n",
            dm_timer, ktime_to_ns(dm_timer->interval), dm_timer->func, dm_timer->periodic);
}

static void dmTimer_worker(struct work_struct *work)
{
    struct DmLinuxTimer *dm_timer = container_of(work, struct DmLinuxTimer, mywork);
    dm_timer->func();
}

static enum hrtimer_restart pseudoHrTimerFunc(struct hrtimer *t)
{
    struct DmLinuxTimer *dm_timer;
    //pr_info("%s\n", __FUNCTION__);
    dm_timer = container_of(t, struct DmLinuxTimer, myhrtimer);
    //printDmLinuxTimer(dm_timer);
    /* Using shared workqueue */
    schedule_work(&dm_timer->mywork);
    if(dm_timer->periodic) {
        ktime_t now = hrtimer_cb_get_time(t);
        hrtimer_forward(t, now, dm_timer->interval);
        return HRTIMER_RESTART;
    }
    return HRTIMER_NORESTART;
}

void* OsStartTimerPri(INT32 lTimeout, void (*pCallback), INT32 bPeriodic, UINT32 priority)
{
    /* Ignore priority for now... */
    struct DmLinuxTimer *dm_timer;
    dm_timer = kzalloc(sizeof(*dm_timer), GFP_KERNEL);
    if(!dm_timer) {
        printk(KERN_ERR "Failed to create timer.\n");
        return 0;
    }
    dm_timer->func = pCallback;
    dm_timer->periodic = !!bPeriodic;
    dm_timer->interval = ktime_set(0, lTimeout * 1000000UL); /* Convert ms to ns */
    INIT_WORK(&dm_timer->mywork, dmTimer_worker);
    // Setup timer
    hrtimer_init(&dm_timer->myhrtimer, CLOCK_MONOTONIC, HRTIMER_MODE_REL);
    dm_timer->myhrtimer.function = &pseudoHrTimerFunc;
    //printDmLinuxTimer(dm_timer);
    hrtimer_start(&dm_timer->myhrtimer, dm_timer->interval, HRTIMER_MODE_REL);
    return dm_timer;
}
/**
 * TODO: Make this function not fail if the timer was running while it was called.
 */
void OsCancelTimer(void *hTimer)
{
    int ret;
    struct DmLinuxTimer *timer = (struct DmLinuxTimer*) (hTimer);
    if(!timer) {
        pr_err("Cannot cancel null timer.\n");
        return;
    }
    ret = hrtimer_cancel(&timer->myhrtimer);
    timer->periodic = false; /* This should stop it after the next run if not already. */
    if(ret) {
        pr_err("Timer was running while attempting to cancel.\n");
    }
}

/**
 * Critical section. Since it doesn't take arguments it needs a global 
 * critical_mutex initialized by OsStart.
 */
void OsEnterCritical(void)
{
    mutex_lock(&critical_mutex);
}
void OsExitCritical(void)
{
    mutex_unlock(&critical_mutex);
}

void* OsAllocMemory(UINT32 byteCnt)
{
    return kzalloc(byteCnt, GFP_KERNEL);
}
void* OsCallocMemory(UINT32 byteCnt, UINT32 elementSize)
{
    return kcalloc(byteCnt, elementSize, GFP_KERNEL);
}

INT32 OsFreeMemory(void *mem)
{
    //TODO: remove this check.
    if(!mem) {
        pr_err("ERROR: ------- Freeing null memory.\n");
    }
    kfree(mem);
    return 0;
}
